<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\Post;

use Exception;
use Throwable;
use Erlage\Photogram\Settings;
use Erlage\Photogram\Tools\Utils;
use Erlage\Photogram\SystemLogger;
use Erlage\Photogram\Tools\Fso\Storage;
use Erlage\Photogram\Tools\Fso\ImageUploader;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Constants\SystemConstants;
use Erlage\Photogram\Data\Common\CommonQueries;
use Erlage\Photogram\Data\Tables\Sys\RestTable;
use Erlage\Photogram\Data\Models\Post\PostModel;
use Erlage\Photogram\Data\Models\User\UserModel;
use Erlage\Photogram\Data\Tables\Post\PostTable;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Models\Post\PostBuilder;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Dtos\Common\DisplayItemDTO;
use Erlage\Photogram\Data\Models\Hashtag\HashtagModel;
use Erlage\Photogram\Data\Models\User\UserModelHelper;
use Erlage\Photogram\Data\Dtos\Post\PostDisplayItemDTO;
use Erlage\Photogram\Data\Models\Hashtag\HashtagFinder;
use Erlage\Photogram\Data\Dtos\Post\PostMetaHashtagsDTO;
use Erlage\Photogram\Data\Models\Hashtag\HashtagBuilder;
use Erlage\Photogram\Data\Dtos\Post\PostDisplayContentDTO;
use Erlage\Photogram\Data\Dtos\Post\PostDisplayUserTagDTO;
use Erlage\Photogram\Data\Dtos\User\UserMetaPushSettingsDTO;
use Erlage\Photogram\Data\Models\Hashtag\HashtagModelHelper;
use Erlage\Photogram\Data\Dtos\Post\PostDisplayContentItemDTO;
use Erlage\Photogram\Data\Models\Notification\NotificationEnum;
use Erlage\Photogram\Data\Dtos\Post\PostDisplayUserTagsOnItemDTO;
use Erlage\Photogram\Data\Models\Hashtag\Post\HashtagPostBuilder;
use Erlage\Photogram\Data\Models\Post\Usertag\PostUserTagBuilder;
use Erlage\Photogram\Data\Models\Notification\NotificationBuilder;
use Erlage\Photogram\Data\Dtos\Notification\NotificationLinkedContentDTO;

final class PostActions extends ExceptionalRequests
{
    /*
    |--------------------------------------------------------------------------
    | add a new post
    |--------------------------------------------------------------------------
    */

    public static function add(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $displayCaptionFromReq = self::$request -> findKey(
                PostTable::DISPLAY_CAPTION,
                RequestTable::PAYLOAD,
                PostTable::TABLE_NAME
            );

            $displayLocationFromReq = self::$request -> findKey(
                PostTable::DISPLAY_LOCATION,
                RequestTable::PAYLOAD,
                PostTable::TABLE_NAME
            );

            $usersTaggedFromReq = self::$request -> getCollection(
                PostTable::DISPLAY_CONTENT,
                RequestTable::PAYLOAD,
                PostTable::TABLE_NAME
            );

            $displayContentItemsFromReq = self::$request -> getCollection(
                PostTable::DISPLAY_CONTENT,
                RestTable::ATTACHEMENTS,
                PostTable::TABLE_NAME
            );

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | create a post builder
            |--------------------------------------------------------------------------
            */

            $postBuilder = (new PostBuilder())
                -> setOwnerUserId(self::$authedUserModel -> getId())
                -> setDisplayCaption($displayCaptionFromReq)
                -> setDisplayLocation($displayLocationFromReq);

            /*
            |--------------------------------------------------------------------------
            | create post content
            |--------------------------------------------------------------------------
            */

            $postDisplayContent = new PostDisplayContentDTO();

            /*
            |--------------------------------------------------------------------------
            | add content
            |--------------------------------------------------------------------------
            */

            $taggedUserIds = array();

            /*
            |--------------------------------------------------------------------------
            | if there are image in post content
            |--------------------------------------------------------------------------
            */

            if (isset($displayContentItemsFromReq['name']))
            {
                /*
                |--------------------------------------------------------------------------
                | for each image in uploaded data
                |--------------------------------------------------------------------------
                */

                foreach ($displayContentItemsFromReq['name'] as $key => $value)
                {
                    /*
                    |--------------------------------------------------------------------------
                    | add tagged users
                    |--------------------------------------------------------------------------
                    */

                    $displayUserTagsOnItem = new PostDisplayUserTagsOnItemDTO();

                    foreach ($usersTaggedFromReq[$key] as $userTagContent)
                    {
                        /*
                        |--------------------------------------------------------------------------
                        | tagged location and user details
                        |--------------------------------------------------------------------------
                        */

                        $displayUserTag = PostDisplayUserTagDTO::fromJsonMap($userTagContent);

                        $displayUserTagsOnItem -> add($displayUserTag);

                        /*
                        |--------------------------------------------------------------------------
                        | add user id to list of users tagged
                        |--------------------------------------------------------------------------
                        */

                        $taggedUserId = $displayUserTag -> getUserId();

                        if ( ! \in_array($taggedUserId, $taggedUserIds))
                        {
                            $taggedUserIds[] = $taggedUserId;
                        }
                    }

                    /*
                    |--------------------------------------------------------------------------
                    | prepare image data map
                    |--------------------------------------------------------------------------
                    */

                    $imageData = array(
                        'size'     => $displayContentItemsFromReq['size'][$key],
                        'type'     => $displayContentItemsFromReq['type'][$key],
                        'name'     => $displayContentItemsFromReq['name'][$key],
                        'error'    => $displayContentItemsFromReq['error'][$key],
                        'tmp_name' => $displayContentItemsFromReq['tmp_name'][$key],
                    );

                    /*
                    |--------------------------------------------------------------------------
                    | create image uploader
                    |--------------------------------------------------------------------------
                    */

                    $host = Settings::getString(ServerConstants::SS_ENUM_STORAGE_DISK_POST_IMAGES);
                    $resolution = Settings::getInt(ServerConstants::SS_INT_COMPRESS_POST_DISPLAY_IMAGE_FILE_RES);
                    $compressedQuality = Settings::getInt(ServerConstants::SS_INT_COMPRESS_POST_DISPLAY_IMAGE_FILE_QUALITY);
                    $maxFileSizeInMegaBytes = Settings::getInt(SystemConstants::SETTING_MAX_POST_DISPLAY_CONTENT_ITEM_FILE_SIZE);
                    $supportedExtensions = Settings::getString(SystemConstants::SETTING_SUPPORTED_POST_DISPLAY_CONTENT_ITEM_FILE_FORMAT);

                    $imageUploader = (new ImageUploader())
                        -> setHost($host)
                        -> setUserId(self::$authedUserModel -> getId())
                        -> setFilespace(DisplayItemDTO::FILESPACE_POST)

                        -> setSaveResolution($resolution)
                        -> setCompressedQuality($compressedQuality)
                        -> setMaxSizeInMegaBytes($maxFileSizeInMegaBytes)
                        -> setSupportedExtensions($supportedExtensions)

                        -> setTemporaryFile($imageData)

                        -> prepare();

                    /*
                    |--------------------------------------------------------------------------
                    | check file format
                    |--------------------------------------------------------------------------
                    */

                    if ( ! $imageUploader -> isValid() || $imageUploader -> isNotSupported())
                    {
                        throw new RequestException(ResponseConstants::D_ERROR_POST_DISPLAY_CONTENT_ITEM_FILE_FORMAT_MSG);
                    }

                    /*
                    |--------------------------------------------------------------------------
                    | check file size
                    |--------------------------------------------------------------------------
                    */

                    if ($imageUploader -> isOverSized())
                    {
                        throw new RequestException(ResponseConstants::D_ERROR_POST_DISPLAY_CONTENT_ITEM_FILE_SIZE_MSG);
                    }

                    /*
                    |--------------------------------------------------------------------------
                    | finish save
                    |--------------------------------------------------------------------------
                    */

                    $uploadHandle = $imageUploader -> process();

                    // check if image upload has succeeded

                    if ( ! $uploadHandle -> processed)
                    {
                        // upload failed, log internally

                        SystemLogger::internalException(new Exception($uploadHandle -> error));

                        // something went wrong

                        throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
                    }

                    // else get DTO object

                    $uploadedImageDTO = $imageUploader -> getDisplayItemDTO();

                    /*
                    |--------------------------------------------------------------------------
                    | add post content item
                    |--------------------------------------------------------------------------
                    */

                    $displayContentItem = (new PostDisplayItemDTO())
                        -> setHost($uploadedImageDTO -> getHost())
                        -> setIdentifier($uploadedImageDTO -> getIdentifier());

                    $postDisplayContentItem = (new PostDisplayContentItemDTO())
                        -> setDisplayItemDTO($displayContentItem)
                        -> setDisplayUserTagsOnItemDTO($displayUserTagsOnItem);

                    $postDisplayContent -> add($postDisplayContentItem);
                }
            }

            /*
            |--------------------------------------------------------------------------
            | add post content
            |--------------------------------------------------------------------------
            */

            $postBuilder -> setDisplayContent($postDisplayContent);

            /*
            |--------------------------------------------------------------------------
            | add hashtags
            |--------------------------------------------------------------------------
            */

            $postMetaHashtags = new PostMetaHashtagsDTO();

            $hashtagsStringLiterals = Utils::getHashtagsFromString($displayCaptionFromReq);

            foreach ($hashtagsStringLiterals as $hashtag)
            {
                $postMetaHashtags -> add($hashtag, '0');
            }

            $postBuilder -> setMetaHashtags($postMetaHashtags);

            /*
            |--------------------------------------------------------------------------
            | save post
            |--------------------------------------------------------------------------
            */

            $postModel = $postBuilder -> dispense();

            $postModel -> save();

            /*
            |--------------------------------------------------------------------------
            | add postusertags
            |--------------------------------------------------------------------------
            */

            foreach ($taggedUserIds as $taggedUserId)
            {
                $postUserTagModel = (new PostUserTagBuilder())
                    -> setTaggedUserId($taggedUserId)
                    -> setTaggedInPostId($postModel -> getId())
                    -> setPostOwnerUserId($postModel -> getOwnerUserId())
                    -> dispense();

                $postUserTagModel -> save();
            }

            /*
            |--------------------------------------------------------------------------
            | get hashtags, create them if doesn't exists
            |--------------------------------------------------------------------------
            */

            /**
             * @var HashtagModel[]
             */
            $hashtagModels = array();

            $finder = (new HashtagFinder());

            $postMetaHashtags -> iteratorPrepare();

            while ($postMetaHashtags -> iteratorForward())
            {
                $hashtagLiteralValue = $postMetaHashtags -> iteratorCurrentHashtagLiteral();

                $finder -> setDisplayText($hashtagLiteralValue) -> find();

                if ($finder -> isFound())
                {
                    $hashtagModel = $finder -> popModelFromResults();
                }
                else
                {
                    $hashtagModel = (new HashtagBuilder()) -> setDisplayText($hashtagLiteralValue) -> dispense();

                    $hashtagModel -> save();
                }

                $hashtagModels[$hashtagLiteralValue] = $hashtagModel;

                $finder -> clean();
            }

            /*
            |--------------------------------------------------------------------------
            | publish post on hashtags
            |--------------------------------------------------------------------------
            */

            foreach ($hashtagModels as $hashtag => $hashtagModel)
            {
                $hashtagPostModel = (new HashtagPostBuilder())
                    -> setHashtagId($hashtagModel -> getId())
                    -> setPostId($postModel -> getId())
                    -> setPostOwnerUserId(self::$authedUserModel -> getId())
                    -> dispense();

                $hashtagPostModel -> save();

                /*
                |--------------------------------------------------------------------------
                | update hashtag cache
                |--------------------------------------------------------------------------
                */

                HashtagModelHelper::updateCachePostsCount($hashtagModel);
            }

            /*
            |--------------------------------------------------------------------------
            | update meta hashtags on new post
            |--------------------------------------------------------------------------
            */

            foreach ($hashtagModels as $hashtagModel)
            {
                $postMetaHashtags -> updateValue($hashtagModel -> getDisplayText(), $hashtagModel -> getId());
            }

            $postModel -> update(array(
                PostTable::META_HASHTAGS => $postMetaHashtags,
            ));

            $postModel -> save();

            /*
            |--------------------------------------------------------------------------
            | add to response
            |--------------------------------------------------------------------------
            */

            self::addToResponse(PostTable::getTableName(), $postModel -> getDataMap());
            self::addToResponse(UserTable::getTableName(), self::$authedUserModel -> getDataMap());

            /*
            |--------------------------------------------------------------------------
            | update user cache
            |--------------------------------------------------------------------------
            */

            UserModelHelper::updateCachePostsCount(self::$authedUserModel);

            /*
            |--------------------------------------------------------------------------
            | notify users who are tagged in the post which just got liked
            |--------------------------------------------------------------------------
            */

            $postDisplayContent = $postModel -> getDisplayContent();

            /*
            |--------------------------------------------------------------------------
            | for all images in post content
            |--------------------------------------------------------------------------
            */

            $postDisplayContent -> iteratorPrepare();

            while ($postDisplayContent -> iteratorForward())
            {
                $displayContentItemDTO = $postDisplayContent -> iteratorCurrentDisplayContentItem();

                /*
                |--------------------------------------------------------------------------
                | get tagged users DTO
                |--------------------------------------------------------------------------
                */

                $taggedUsersDTO = $displayContentItemDTO -> getDisplayUserTagsOnItemDTO();

                /*
                |--------------------------------------------------------------------------
                | for all tagged users
                |--------------------------------------------------------------------------
                */

                $taggedUsersDTO -> iteratorPrepare();

                while ($taggedUsersDTO -> iteratorForward())
                {
                    $taggedUserDTO = $taggedUsersDTO -> iteratorCurrentDisplayUserTag();

                    /*
                    |--------------------------------------------------------------------------
                    | if tagged user is post owner then don't send them a notification
                    |--------------------------------------------------------------------------
                    */

                    if (self::$authedUserModel -> getId() == $taggedUserDTO -> getUserId())
                    {
                        continue;
                    }

                    /*
                    |--------------------------------------------------------------------------
                    | get tagged user model
                    |--------------------------------------------------------------------------
                    */

                    $taggedUserModel = UserModel::findFromId_noException($taggedUserDTO -> getUserId());

                    if ($taggedUserModel -> isModel())
                    {
                        $pushSettings = $taggedUserModel -> getMetaPushSettings();

                        /*
                        |--------------------------------------------------------------------------
                        | if tagged user has opted in for notifications
                        |--------------------------------------------------------------------------
                        */

                        switch ($pushSettings -> getPhotosOfYou())
                        {
                            case UserMetaPushSettingsDTO::VALUE_OFF:
                                $notifyUser = false;

                            break;

                            case UserMetaPushSettingsDTO::VALUE_FROM_PEOPLE_I_FOLLOW:
                                $notifyUser = CommonQueries::isFollowing(
                                    $taggedUserModel -> getId(),
                                    self::$authedUserModel -> getId()
                                );

                            break;

                            default: $notifyUser = true;
                        }

                        if ($notifyUser)
                        {
                            /*
                            |--------------------------------------------------------------------------
                            | create and dispatch a new notification
                            |--------------------------------------------------------------------------
                            */

                            $notificationContent = new NotificationLinkedContentDTO();

                            $notificationContent -> linkUserId(self::$authedUserModel -> getId());
                            $notificationContent -> linkPostId($postModel -> getId());

                            $notificationModel = (new NotificationBuilder())
                                -> setToUserId($taggedUserModel -> getId())
                                -> setTargetContentId($postModel -> getId())
                                -> setMetaType(NotificationEnum::META_TYPE_PHOTO_OF_YOU)
                                -> setLinkedContent($notificationContent)
                                -> dispense();

                            $notificationModel -> save();
                        }
                    }
                }
            }
        });
    }

    public static function remove(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $postIdFromReq = self::$request -> findKey(PostTable::ID, RequestTable::PAYLOAD, PostTable::TABLE_NAME);

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $postIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | ensure post exists
            |--------------------------------------------------------------------------
            */

            $targetPostModel = PostModel::findFromId_throwException($postIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | ensure ownership
            |--------------------------------------------------------------------------
            */

            if ($targetPostModel -> getOwnerUserId() != self::$authedUserModel -> getId())
            {
                throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | object references for delete
            |--------------------------------------------------------------------------
            */

            $obsolutePostDisplayContent = $targetPostModel -> getDisplayContent();

            /*
            |--------------------------------------------------------------------------
            | delete post
            |--------------------------------------------------------------------------
            */

            $targetPostModel -> delete();

            /*
            |--------------------------------------------------------------------------
            | clean objects from storage
            |--------------------------------------------------------------------------
            */

            try
            {
                $obsolutePostDisplayContent -> iteratorPrepare();

                while ($obsolutePostDisplayContent -> iteratorForward())
                {
                    $displayContentItemDTO = $obsolutePostDisplayContent -> iteratorCurrentDisplayContentItem();

                    $displayItemDTO = $displayContentItemDTO -> getDisplayItemDTO();

                    Storage::disk($displayItemDTO -> getHost()) -> deleteDTO($displayItemDTO);
                }
            }
            catch (Throwable $e)
            {
                SystemLogger::internalException($e);
            }

            /*
            |--------------------------------------------------------------------------
            | update user post count
            |--------------------------------------------------------------------------
            */

            UserModelHelper::updateCachePostsCount(self::$authedUserModel);
        });
    }
}
